#!/usr/bin/env node

/**
 * TTS Vietnamese Server với Google Cloud Text-to-Speech API
 * =========================================================
 * Version: 5.0.0 - Giọng đọc tiếng Việt thật
 * 
 * THAY ĐỔI API KEY TẠI: /opt/tts-server/.env
 * Dòng: GOOGLE_CLOUD_API_KEY=your_google_cloud_api_key_here
 */

'use strict';

// ===== IMPORTS =====
const express = require('express');
const cors = require('cors');
const rateLimit = require('express-rate-limit');
const helmet = require('helmet');
const compression = require('compression');
const fs = require('fs').promises;
const path = require('path');
const crypto = require('crypto');
const axios = require('axios');
require('dotenv').config();

// ===== CONFIGURATION =====
const CONFIG = {
    // ⚠️ THAY ĐỔI API KEY TẠI FILE .env - DÒNG GOOGLE_CLOUD_API_KEY
    GOOGLE_CLOUD_API_KEY: process.env.GOOGLE_CLOUD_API_KEY || '',
    ALLOWED_ORIGINS: process.env.ALLOWED_ORIGINS || 'http://localhost,http://127.0.0.1,http://vip123.me,http://35.240.230.211,https://vip123.me,https://35.240.230.211',
    PORT: parseInt(process.env.PORT) || 3000,
    NODE_ENV: process.env.NODE_ENV || 'production',
    MAX_TEXT_LENGTH: parseInt(process.env.MAX_TEXT_LENGTH) || 5000,
    RATE_LIMIT_WINDOW: parseInt(process.env.RATE_LIMIT_WINDOW_MS) || 15 * 60 * 1000,
    RATE_LIMIT_MAX: parseInt(process.env.RATE_LIMIT_MAX) || 50,
    TTS_RATE_LIMIT_WINDOW: parseInt(process.env.TTS_RATE_LIMIT_WINDOW_MS) || 10 * 60 * 1000,
    TTS_RATE_LIMIT_MAX: parseInt(process.env.TTS_RATE_LIMIT_MAX) || 10,
    CLEANUP_INTERVAL: parseInt(process.env.CLEANUP_INTERVAL_MS) || 60 * 60 * 1000,
    FILE_MAX_AGE: parseInt(process.env.FILE_MAX_AGE_MS) || 2 * 60 * 60 * 1000,
};

// ===== GLOBALS =====
const app = express();
const UPLOAD_DIR = path.join(__dirname, 'public', 'audio');
let cleanupTimer;

// ===== TRUST PROXY =====
app.set('trust proxy', 1);

// ===== UTILITY FUNCTIONS =====
const logger = {
    info: (msg, ...args) => console.log(`[${new Date().toISOString()}] [INFO] ${msg}`, ...args),
    warn: (msg, ...args) => console.log(`[${new Date().toISOString()}] [WARN] ${msg}`, ...args),
    error: (msg, ...args) => console.error(`[${new Date().toISOString()}] [ERROR] ${msg}`, ...args),
};

const formatBytes = (bytes) => {
    if (bytes === 0) return '0 B';
    const k = 1024;
    const sizes = ['B', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return `${parseFloat((bytes / Math.pow(k, i)).toFixed(2))} ${sizes[i]}`;
};

const generateSecureFilename = () => {
    const timestamp = Date.now();
    const random = crypto.randomBytes(12).toString('hex');
    return `tts_${timestamp}_${random}.wav`;
};

// ===== DIRECTORY MANAGEMENT =====
const ensureDirectoryExists = async (dirPath) => {
    try {
        await fs.access(dirPath);
    } catch {
        await fs.mkdir(dirPath, { recursive: true, mode: 0o755 });
        logger.info(`Created directory: ${dirPath}`);
    }
};

const cleanupOldFiles = async () => {
    try {
        await ensureDirectoryExists(UPLOAD_DIR);
        const files = await fs.readdir(UPLOAD_DIR);
        const now = Date.now();
        let deletedCount = 0;
        
        for (const file of files) {
            if (!file.endsWith('.wav')) continue;
            const filePath = path.join(UPLOAD_DIR, file);
            try {
                const stats = await fs.stat(filePath);
                if (now - stats.mtime.getTime() > CONFIG.FILE_MAX_AGE) {
                    await fs.unlink(filePath);
                    deletedCount++;
                }
            } catch (error) {
                logger.warn(`Failed to process file ${file}:`, error.message);
            }
        }
        
        if (deletedCount > 0) {
            logger.info(`Cleaned up ${deletedCount} old audio files`);
        }
    } catch (error) {
        logger.error('Cleanup failed:', error.message);
    }
};

// ===== VALIDATION =====
const validateText = (text) => {
    if (!text || typeof text !== 'string') {
        throw new Error('Text không hợp lệ');
    }
    const cleaned = text.trim().replace(/\s+/g, ' ');
    if (cleaned.length === 0) {
        throw new Error('Text không được để trống');
    }
    if (cleaned.length > CONFIG.MAX_TEXT_LENGTH) {
        throw new Error(`Text quá dài (tối đa ${CONFIG.MAX_TEXT_LENGTH} ký tự)`);
    }
    return cleaned;
};

// ===== GOOGLE CLOUD TTS FUNCTION =====
const generateGoogleTTS = async (text, voiceName = 'vi-VN-Standard-A') => {
    if (!CONFIG.GOOGLE_CLOUD_API_KEY) {
        throw new Error('⚠️ GOOGLE_CLOUD_API_KEY chưa được cấu hình trong file .env');
    }

    const url = `https://texttospeech.googleapis.com/v1/text:synthesize?key=${CONFIG.GOOGLE_CLOUD_API_KEY}`;
    
    const requestBody = {
        input: {
            text: text
        },
        voice: {
            languageCode: 'vi-VN',
            name: voiceName,
            ssmlGender: 'FEMALE'
        },
        audioConfig: {
            audioEncoding: 'LINEAR16',
            sampleRateHertz: 24000,
            speakingRate: 1.0,
            pitch: 0.0,
            volumeGainDb: 0.0
        }
    };

    logger.info(`🔗 Calling Google Cloud TTS API: ${text.substring(0, 50)}... (Voice: ${voiceName})`);

    try {
        const response = await axios.post(url, requestBody, {
            headers: {
                'Content-Type': 'application/json',
                'User-Agent': 'TTS-Vietnamese-Server/5.0.0'
            },
            timeout: 30000
        });

        if (!response.data.audioContent) {
            throw new Error('❌ Không nhận được dữ liệu audio từ Google Cloud TTS');
        }

        // Chuyển base64 thành buffer
        const audioBuffer = Buffer.from(response.data.audioContent, 'base64');
        logger.info(`✅ Google TTS completed: ${formatBytes(audioBuffer.length)}`);
        
        return audioBuffer;

    } catch (error) {
        if (error.response) {
            const status = error.response.status;
            const errorData = error.response.data;
            
            logger.error(`❌ Google Cloud TTS API error: ${status} - ${JSON.stringify(errorData)}`);
            
            if (status === 401 || status === 403) {
                throw new Error('❌ API Key không hợp lệ hoặc không có quyền truy cập. Kiểm tra GOOGLE_CLOUD_API_KEY trong .env');
            } else if (status === 429) {
                throw new Error('❌ Đã vượt quá giới hạn API. Vui lòng thử lại sau.');
            } else {
                throw new Error(`❌ Google Cloud TTS API error: ${status} ${errorData.error?.message || 'Unknown error'}`);
            }
        } else {
            logger.error(`❌ Network error: ${error.message}`);
            throw new Error(`❌ Lỗi kết nối tới Google Cloud TTS: ${error.message}`);
        }
    }
};

// ===== DEMO AUDIO FALLBACK =====
const generateDemoAudio = async (text) => {
    logger.warn('🔄 Using demo audio fallback (Google Cloud TTS not available)');
    
    const sampleRate = 44100;
    const duration = Math.max(2, text.length * 0.15);
    const numSamples = Math.floor(sampleRate * duration);
    
    const buffer = Buffer.alloc(44 + numSamples * 2);
    
    // WAV header
    buffer.write('RIFF', 0);
    buffer.writeUInt32LE(36 + numSamples * 2, 4);
    buffer.write('WAVE', 8);
    buffer.write('fmt ', 12);
    buffer.writeUInt32LE(16, 16);
    buffer.writeUInt16LE(1, 20);
    buffer.writeUInt16LE(1, 22);
    buffer.writeUInt32LE(sampleRate, 24);
    buffer.writeUInt32LE(sampleRate * 2, 28);
    buffer.writeUInt16LE(2, 32);
    buffer.writeUInt16LE(16, 34);
    buffer.write('data', 36);
    buffer.writeUInt32LE(numSamples * 2, 40);
    
    // Tạo âm thanh demo với speech-like patterns
    for (let i = 0; i < numSamples; i++) {
        const t = i / sampleRate;
        const envelope = Math.min(1, Math.min(t * 2, (duration - t) * 2));
        
        // Tạo pattern giống giọng nói hơn
        const freq1 = 200 + Math.sin(t * 10) * 50; // Fundamental frequency
        const freq2 = 400 + Math.sin(t * 15) * 30; // First harmonic
        const freq3 = 600 + Math.sin(t * 20) * 20; // Second harmonic
        
        const sample1 = Math.sin(2 * Math.PI * freq1 * t) * 0.4;
        const sample2 = Math.sin(2 * Math.PI * freq2 * t) * 0.2;
        const sample3 = Math.sin(2 * Math.PI * freq3 * t) * 0.1;
        
        // Thêm noise để giống giọng nói hơn
        const noise = (Math.random() - 0.5) * 0.05;
        
        const combinedSample = (sample1 + sample2 + sample3 + noise) * envelope * 32767;
        buffer.writeInt16LE(Math.floor(combinedSample), 44 + i * 2);
    }
    
    await new Promise(resolve => setTimeout(resolve, 1000));
    return buffer;
};

// ===== MIDDLEWARE =====
app.use(helmet());
app.use(compression({ level: 6, threshold: 1024 }));
app.use(express.json({ limit: '2mb' }));
app.use(express.urlencoded({ extended: true, limit: '2mb' }));

const createRateLimit = (windowMs, max, message) => {
    return rateLimit({
        windowMs,
        max,
        message: { success: false, error: message, retryAfter: Math.ceil(windowMs / 1000) },
        standardHeaders: true,
        legacyHeaders: false,
        skip: (req) => req.path === '/health',
    });
};

app.use(createRateLimit(CONFIG.RATE_LIMIT_WINDOW, CONFIG.RATE_LIMIT_MAX, 'Quá nhiều yêu cầu từ IP này'));

// ===== CORS =====
app.use(cors({
    origin: (origin, callback) => {
        const allowedOrigins = CONFIG.ALLOWED_ORIGINS.split(',').map(o => o.trim());
        if (!origin) return callback(null, true);
        if (allowedOrigins.includes(origin)) {
            callback(null, true);
        } else {
            logger.warn(`Blocked CORS request from: ${origin}`);
            callback(new Error('Not allowed by CORS'));
        }
    },
    credentials: true,
    methods: ['GET', 'POST', 'OPTIONS'],
    allowedHeaders: ['Content-Type', 'Authorization', 'X-Requested-With'],
    maxAge: 86400
}));

// ===== STATIC FILES =====
app.use('/audio', express.static(path.join(__dirname, 'public', 'audio'), {
    maxAge: '1h',
    etag: true,
    lastModified: true,
    setHeaders: (res, filePath) => {
        res.setHeader('X-Content-Type-Options', 'nosniff');
        res.setHeader('Cache-Control', 'public, max-age=3600, immutable');
    }
}));

app.use(express.static(path.join(__dirname, 'public'), {
    maxAge: '1d',
    etag: true,
    index: 'index.html'
}));

// ===== API ENDPOINTS =====
app.get('/health', (req, res) => {
    const uptime = process.uptime();
    const memUsage = process.memoryUsage();
    const hasApiKey = !!CONFIG.GOOGLE_CLOUD_API_KEY;
    
    res.json({
        status: 'healthy',
        timestamp: new Date().toISOString(),
        uptime: Math.floor(uptime),
        memory: {
            rss: formatBytes(memUsage.rss),
            heapUsed: formatBytes(memUsage.heapUsed),
            heapTotal: formatBytes(memUsage.heapTotal),
            external: formatBytes(memUsage.external)
        },
        version: '5.0.0',
        environment: CONFIG.NODE_ENV,
        nodeVersion: process.version,
        platform: process.platform,
        arch: process.arch,
        ttsEngine: hasApiKey ? 'Google Cloud Text-to-Speech' : 'Demo Audio Generator',
        apiKeyConfigured: hasApiKey
    });
});

app.get('/api/voices', (req, res) => {
    const hasApiKey = !!CONFIG.GOOGLE_CLOUD_API_KEY;
    
    const voices = hasApiKey ? [
        { name: 'vi-VN-Standard-A', description: 'Giọng nữ miền Nam chuẩn', gender: 'female', accent: 'southern', recommended: true },
        { name: 'vi-VN-Standard-B', description: 'Giọng nam miền Nam chuẩn', gender: 'male', accent: 'southern' },
        { name: 'vi-VN-Standard-C', description: 'Giọng nữ miền Bắc chuẩn', gender: 'female', accent: 'northern' },
        { name: 'vi-VN-Standard-D', description: 'Giọng nam miền Bắc chuẩn', gender: 'male', accent: 'northern' },
        { name: 'vi-VN-Wavenet-A', description: 'Giọng nữ miền Nam WaveNet (cao cấp)', gender: 'female', accent: 'southern', premium: true },
        { name: 'vi-VN-Wavenet-B', description: 'Giọng nam miền Nam WaveNet (cao cấp)', gender: 'male', accent: 'southern', premium: true },
        { name: 'vi-VN-Wavenet-C', description: 'Giọng nữ miền Bắc WaveNet (cao cấp)', gender: 'female', accent: 'northern', premium: true },
        { name: 'vi-VN-Wavenet-D', description: 'Giọng nam miền Bắc WaveNet (cao cấp)', gender: 'male', accent: 'northern', premium: true }
    ] : [
        { name: 'demo', description: 'Demo Audio Generator (Cần cấu hình API Key)', gender: 'synthetic', accent: 'demo' }
    ];
    
    res.json({ 
        success: true, 
        voices, 
        total: voices.length,
        recommended: voices.filter(v => v.recommended).map(v => v.name),
        ttsEngine: hasApiKey ? 'Google Cloud Text-to-Speech' : 'Demo Audio Generator',
        apiKeyConfigured: hasApiKey,
        note: hasApiKey ? 'Google Cloud TTS đã sẵn sàng' : '⚠️ Cần cấu hình GOOGLE_CLOUD_API_KEY trong file .env'
    });
});

app.post('/api/tts',
    createRateLimit(CONFIG.TTS_RATE_LIMIT_WINDOW, CONFIG.TTS_RATE_LIMIT_MAX, 'Quá nhiều yêu cầu TTS'),
    async (req, res) => {
        const startTime = Date.now();
        const requestId = crypto.randomBytes(8).toString('hex');
        
        try {
            const { text, voice = 'vi-VN-Standard-A' } = req.body;
            const cleanText = validateText(text);
            
            logger.info(`🎙️ [${requestId}] TTS Request: ${cleanText.substring(0, 50)}... (Voice: ${voice})`);
            
            let audioBuffer;
            let engineUsed;
            
            // Thử dùng Google Cloud TTS trước, fallback về demo nếu lỗi
            try {
                audioBuffer = await generateGoogleTTS(cleanText, voice);
                engineUsed = 'Google Cloud Text-to-Speech';
            } catch (googleError) {
                logger.warn(`⚠️ Google Cloud TTS failed: ${googleError.message}`);
                logger.info('🔄 Falling back to demo audio');
                audioBuffer = await generateDemoAudio(cleanText);
                engineUsed = 'Demo Audio Generator (Fallback)';
            }
            
            await ensureDirectoryExists(UPLOAD_DIR);
            const fileName = generateSecureFilename();
            const filePath = path.join(UPLOAD_DIR, fileName);
            
            await fs.writeFile(filePath, audioBuffer, { mode: 0o644 });
            
            const processingTime = Date.now() - startTime;
            logger.info(`✅ [${requestId}] TTS completed in ${processingTime}ms using ${engineUsed}`);
            
            res.json({
                success: true,
                audioUrl: `/audio/${fileName}`,
                message: 'Audio generated successfully',
                metadata: {
                    requestId,
                    fileSize: audioBuffer.length,
                    fileSizeFormatted: formatBytes(audioBuffer.length),
                    processingTime: `${processingTime}ms`,
                    textLength: cleanText.length,
                    voice: voice,
                    timestamp: new Date().toISOString(),
                    ttsEngine: engineUsed,
                    sampleRate: engineUsed.includes('Google') ? '24kHz' : '44.1kHz',
                    bitDepth: '16-bit',
                    channels: 'Mono'
                }
            });
            
        } catch (error) {
            const processingTime = Date.now() - startTime;
            logger.error(`❌ [${requestId}] TTS failed in ${processingTime}ms:`, error.message);
            
            let statusCode = 500;
            let errorMessage = 'Internal server error';
            
            if (error.message.includes('quá dài') || error.message.includes('không hợp lệ') || error.message.includes('trống')) {
                statusCode = 400;
                errorMessage = error.message;
            } else if (error.message.includes('API Key') || error.message.includes('401') || error.message.includes('403')) {
                statusCode = 401;
                errorMessage = 'API Key không hợp lệ. Vui lòng kiểm tra cấu hình.';
            } else if (error.message.includes('429') || error.message.includes('quota')) {
                statusCode = 429;
                errorMessage = 'Đã vượt quá giới hạn API. Vui lòng thử lại sau.';
            }
            
            res.status(statusCode).json({
                success: false,
                error: errorMessage,
                requestId,
                processingTime: `${processingTime}ms`,
                note: 'Kiểm tra GOOGLE_CLOUD_API_KEY trong file .env nếu lỗi liên tục'
            });
        }
    }
);

// ===== ERROR HANDLING =====
app.use((req, res) => {
    res.status(404).json({ 
        success: false, 
        error: 'Endpoint not found' 
    });
});

app.use((err, req, res, next) => {
    logger.error('Unhandled error:', err.message);
    res.status(err.status || 500).json({
        success: false,
        error: err.message || 'Internal server error'
    });
});

// ===== SERVER STARTUP =====
const startServer = async () => {
    try {
        await ensureDirectoryExists(UPLOAD_DIR);
        
        // Kiểm tra API Key
        const hasApiKey = !!CONFIG.GOOGLE_CLOUD_API_KEY;
        if (!hasApiKey) {
            logger.warn('⚠️ GOOGLE_CLOUD_API_KEY chưa được cấu hình trong file .env');
            logger.warn('⚠️ Server sẽ chạy ở chế độ demo. Để có giọng đọc thật:');
            logger.warn('   1. Tạo Google Cloud project và enable Text-to-Speech API');
            logger.warn('   2. Tạo API key từ Google Cloud Console');
            logger.warn('   3. Thêm vào file .env: GOOGLE_CLOUD_API_KEY=your_api_key_here');
            logger.warn('   4. Restart server: systemctl restart tts-server');
        }
        
        // Khởi động cleanup timer
        cleanupTimer = setInterval(cleanupOldFiles, CONFIG.CLEANUP_INTERVAL);
        cleanupOldFiles();
        
        app.listen(CONFIG.PORT, '127.0.0.1', () => {
            logger.info(`
==========================================
🚀 TTS Vietnamese Server started!
📋 Version: 5.0.0 (Google Cloud TTS)
🔗 URL: http://127.0.0.1:${CONFIG.PORT}
🌐 Environment: ${CONFIG.NODE_ENV}
⚙️  Memory: ${formatBytes(process.memoryUsage().rss)}
🎵 TTS Engine: ${hasApiKey ? 'Google Cloud Text-to-Speech' : 'Demo Mode'}
🔑 API Key: ${hasApiKey ? 'Configured ✅' : 'Not configured ⚠️'}
📝 Config file: /opt/tts-server/.env
==========================================
            `);
        });
        
    } catch (error) {
        logger.error('Server startup failed:', error.message);
        process.exit(1);
    }
};

// Graceful shutdown
const gracefulShutdown = () => {
    logger.info('Received shutdown signal. Closing server...');
    if (cleanupTimer) clearInterval(cleanupTimer);
    logger.info('Cleanup complete. Exiting...');
    process.exit(0);
};

process.on('SIGTERM', gracefulShutdown);
process.on('SIGINT', gracefulShutdown);

startServer();
